/**
* \file: automounter_api_ctrl.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#ifndef AUTOMOUNTER_API_CTRL_H_
#define AUTOMOUNTER_API_CTRL_H_

#include "automounter_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief defines a callback function used to signal the application that a umount or remount request has been
 * processed completely
 * \param request_id the id assigned to the request
 * \param result the error code of the request. In case of no errors, RESULT_OK is set here.
 * \param error_msg an error message that belongs to the error code passed with the result parameter
 */
typedef void (* request_done_callback_t)(int request_id, error_code_t result, const char *error_msg);

/**
 * \brief Call to request umounting all partitions of a device.
 *
 * The call is processed asynchronously. The given callback function is called ones the automounter finished processing
 * the request.
 *
 * \param device_id the id of the device
 * \param done_callback_func the callback function to be called by the shared library as soon as the automounter has
 * finished processing the request. NULL can be passed if the application is not interested in the event.
 * \param request_id this id is passed as parameter to the callback function and can be used to identify the request
 *
 * \return This functions returns RESULT_OK in case of no error. Following error codes might be returned in cased of
 * 		   failures:
 * 		   - RESULT_NORESOURCES: 	The function call failed due to resource problems.
 * 		   - RESULT_INVALID: 		The device is not known to the automounter
 * 		   - RESULT_NOT_MOUNTED:	The device has been unmounted before.
 */
error_code_t automounter_api_umount_device(const char *device_id,
		int request_id, request_done_callback_t done_callback_func);

/**
 * \brief Call to request umounting a partition identified by the given partition id.
 *
 * The call is processed asynchronously. The given callback function is called ones the automounter finished processing
 * the request.
 *
 * \param partition_id the id of the partition
 * \param done_callback_func the callback function to be called by the shared library as soon as the automounter has
 * finished processing the request. NULL can be passed if the application is not interested in the event.
 * \param request_id this id is passed as parameter to the callback function and can be used to identify the request
 *
 * \return This functions returns RESULT_OK in case of no error. Following error codes might be returned in cased of
 * 		   failures:
 * 		   - RESULT_NORESOURCES: 	The function call failed due to resource problems.
 * 		   - RESULT_INVALID: 		The partition is not known to the automounter
 * 		   - RESULT_NOT_MOUNTED:	The partition is not mounted.
 */
error_code_t automounter_api_umount_partition_by_id(const char *partition_id,
		int request_id, request_done_callback_t done_callback_func);

/**
 * \brief Call to request umounting a partition identified by the given mount point.
 *
 * The call is processed asynchronously. The given callback function is called ones the automounter finished processing
 * the request.
 *
 * \param mount_point the mount point where the partition is mounted
 * \param done_callback_func the callback function to be called by the shared library as soon as the automounter has
 * finished processing the request. NULL can be passed if the application is not interested in the event.
 * \param request_id this id is passed as parameter to the callback function and can be used to identify the request
 *
 * \return This functions returns RESULT_OK in case of no error. Following error codes might be returned in cased of
 * 		   failures:
 * 		   - RESULT_NORESOURCES: 	The function call failed due to resource problems.
 * 		   - RESULT_INVALID: 		The mount point is not known to the automounter
 */
error_code_t automounter_api_umount_partition_by_mountpoint(const char *mount_point,
		int request_id, request_done_callback_t done_callback_func);

/**
 * \brief Call to request remounting a partition identified by the given partition id with a different set of mount options.
 *
 * The call is processed asynchronously. The given callback function is called ones the automounter finished processing
 * the request.
 *
 * \param partition_id the id of the partition
 * \param options the mount options for the remount
 * \param done_callback_func the callback function to be called by the shared library as soon as the automounter has
 * finished processing the request. NULL can be passed if the application is not interested in the event.
 * \param request_id this id is passed as parameter to the callback function and can be used to identify the request
 *
 * \return This functions returns RESULT_OK in case of no error. Following error codes might be returned in cased of
 * 		   failures:
 * 		   - RESULT_NORESOURCES: 	The function call failed due to resource problems.
 * 		   - RESULT_INVALID: 		The partition is not known to the automounter
 * 		   - RESULT_MOUNT_ERR: 		The automounter was unable to remount the partition with the given set of options
 * 		   - RESULT_NOT_MOUNTED:	The partition is not mounted.
 */
error_code_t automounter_api_remount_partition_by_id(const char *partition_id,const char *options,
		int request_id, request_done_callback_t done_callback_func);

/**
 * \brief Call to request remounting a partition identified by the given mount point with a different set of mount options.
 *
 * The call is processed asynchronously. The given callback function is called ones the automounter finished processing
 * the request.
 *
 * \param mount_point the mount point where the partition is mounted
 * \param options the mount options for the remount
 * \param done_callback_func the callback function to be called by the shared library as soon as the automounter has
 * finished processing the request. NULL can be passed if the application is not interested in the event.
 * \param request_id this id is passed as parameter to the callback function and can be used to identify the request
 *
 * \return This functions returns RESULT_OK in case of no error. Following error codes might be returned in cased of
 * 		   failures:
 * 		   - RESULT_NORESOURCES: 	The function call failed due to resource problems.
 * 		   - RESULT_INVALID: 		The mount point is not known to the automounter
 * 		   - RESULT_MOUNT_ERR: 		The automounter was unable to remount the partition with the given set of options
 */
error_code_t automounter_api_remount_partition_by_mountpoint(const char *mount_point,const char *options,
		int request_id, request_done_callback_t done_callback_func);

#ifdef __cplusplus
}
#endif

#endif
